*-----------------------------------------------------------
* Title      :  Interactive Assembly Project 
* Written by :  Josh Wade
* Date       :  10/7/2018
* Description:  Runs the game Triangle Jumping League II: Pointy Nights
*               In TJLII: PN, the player is a small square that jumps from platform to platform
*               To win, the player must move to a series of randomly selected sigils
*               Each sigil is worth 1 point
*               After all sigils have been collected, the player is shown how fast they were able to get all the sigils     
*               Left and Right arrow keys to move, Space to jump           
*-----------------------------------------------------------  
; Stores all registers for movem operations
ALL_REG                 REG     D0-D7/A0-A6

* BITMAP PRINTING EQUS *
; Signature name for the bitmap
BMP_SIGNATURE   EQU 'BM'

; Sizes to skip through sections of the header
HEADER_SKIP_ARRAY_OFFSET    EQU 10
HEADER_SKIP_WIDTH           EQU 18
HEADER_SKIP_HEIGHT          EQU 22
HEADER_SKIP_BITS            EQU 28

; Parameter offset for navigating the stack in the chunk printing subroutine
OFFSET_IMAGE_FILE        EQU 4      ; Pointer to the image file
OFFSET_CHUNK_CORNER_X    EQU 8      ; X-coordinate of the top-left corner of the chunk
OFFSET_CHUNK_CORNER_Y    EQU 12     ; Y-coordinate of the top-left corner of the chunk
OFFSET_CHUNK_WIDTH       EQU 16     ; Width of the chunk in pixels
OFFSET_CHUNK_HEIGHT      EQU 20     ; Height of the chunk in pixels
OFFSET_DISPLAY_CORNER_X  EQU 24     ; X-coordinate of the top-left corner of the display
OFFSET_DISPLAY_CORNER_Y  EQU 28     ; Y-coordinate of the top-left corner of the display

; Variable for resetting the stack after the chunk print subroutine
RESET_STACK_OFFSET       EQU 28

; Parameter values
; Edit values here to change how the image is outputted
PARAMETER_CHUNK_CORNER_X    EQU 0
PARAMETER_CHUNK_CORNER_Y    EQU 0
PARAMETER_CHUNK_WIDTH       EQU 512
PARAMETER_CHUNK_HEIGHT      EQU 384
PARAMETER_DISPLAY_CORNER_X  EQU 0
PARAMETER_DISPLAY_CORNER_Y  EQU 0

* SIGIL EQUS *
; Track the sigils, which are the pickups in this game
SIGIL_COUNT                         EQU 5
SIGIL_SIZE                          EQU 102
PARAMETER_SIGIL_CORNER_X            EQU 0
PARAMETER_SIGIL_CORNER_Y            EQU 393
PARAMETER_SIGIL_WIDTH               EQU 102
PARAMETER_SIGIL_HEIGHT              EQU 63
PARAMETER_SIGIL_DISPLAY_CORNER_X    EQU 0
PARAMETER_SIGIL_DISPLAY_CORNER_Y    EQU 393
SIGIL_COLOR_GREEN                   EQU $11FFB8
SIGIL_COLOR_ORANGE                  EQU $12B1FE
SIGIL_COLOR_TEAL                    EQU $F1FD13
SIGIL_COLOR_RED                     EQU $2E29E7
SIGIL_COLOR_PURPLE                  EQU $FC14EB

; Bit counts for different color types    
BIT_COLOR_24    EQU 24
BIT_COLOR_32    EQU 32

; Values for console window size
; The Fullscreen option is not used in the final build, but has been left for posterity
OUTPUT_RESOLUTION_WIDTH         EQU 512
OUTPUT_RESOLUTION_HEIGHT        EQU 384
OUTPUT_RESOLUTION_FULLSCREEN    EQU 2

* PLAYER EQUS*
; Used to determine how fast the player object moves in different situations
GRAVITY_ACCELERATION_VALUE  EQU     $4
JUMP_VELOCITY_VALUE         EQU     $18
WALK_VELOCITY_VALUE         EQU     $10
TIMESCALE_VALUE             EQU     $32     ; Used for fixed point math 
BEGIN_FALL                  EQU     $2F4
PLAYER_SIZE                 EQU     50
PLAYER_POSITION             EQU     250
; These dictate how far up, right, and left the player can go
UPPER_BOUND                 EQU     1
LEFT_BOUND                  EQU     0
RIGHT_BOUND                 EQU     462
PLAYER_FILL_COLOR           EQU     $00FFFFFF
PLAYER_OUTLINE_COLOR        EQU     $00FFFFFF
; Size of every item in our collider list 
COLLIDER_INFO_SIZE          EQU     10  

* TRAP CODE EQUS * 
; Task values for trap codes
TASK_PRINT_NUMBER       EQU     3
TASK_GET_TIME           EQU     8
CLEAR_SCREEN_VALUE      EQU     11
RESET_SCREEN            EQU     $FF00
TASK_PRINT_MESSAGE      EQU     13
TASK_READ_INPUT         EQU     19
TASK_OUTPUT_RESOLUTION  EQU     33
TASK_PEN_COLOR          EQU     80
TASK_FILL_COLOR         EQU     81
TASK_DRAW_PIXEL         EQU     82
TASK_DRAW_LINE          EQU     84
TASK_DRAW_SQUARE        EQU     87
TASK_SET_DRAW_MODE      EQU     92
TASK_DRAW_MODE_BUFFER   EQU     17    
TASK_UPDATE_BUFFER      EQU     94

* SEVEN SEGMENT EQUS *
; Equs for displaying the score as a seven segment display
SEG_COLOR_RED           EQU $3333CC
SEG_COLOR_BLACK         EQU $111144
SEG_HORIZONTAL_OFFSET   EQU 20
SEG_VERTICAL_OFFSET     EQU 400
SEG_STARTING_HORIZONTAL EQU 200
SEG_SCALING             EQU 4

* SCORE EQUS *
; Dictate how the player is scored
MAXIMUM_SCORE           EQU 32
SCORE_VALUE             EQU 1

; Input values for player control
; Looks for space bar, left arrow, and right arrow input 
KEY_SPLR_INPUT          EQU     $202527

 
Subroutines   ORG     $0050
    

* Subroutine ConvertEndianWord *       
; Converts the word in d7's endianness
ConvertEndianWord
    ror.w   #8, d7 
    rts


* Subroutine ConvertEndianLong *    
; Converts the long in d7's endianness
ConvertEndianLong
    ror.w   #8, d7 
    swap    d7 
    ror.w   #8, d7 
    rts
    

* Subroutine InitBackgroundMetadata *
; Read the metadata for the BMP
; Includes the label SkipOffset
InitBackgroundMetadata
    ; Reads the file signature 
    ; Throws an error if it's not a BMP file
    cmp.w   #BMP_SIGNATURE, (a0)
    bne     ErrorMessage
    
    ; Reads the pixel array offset
    ; This is the offset from the image pointer that starts the pixel array of the image
    move.l  HEADER_SKIP_ARRAY_OFFSET(a0), d7
    jsr     ConvertEndianLong
    move.l  d7, (pixel_array_offset)

    ; Reads the image width in pixels
    move.l  HEADER_SKIP_WIDTH(a0), d7
    jsr     ConvertEndianLong
    move.l  d7, (width)
    
    ; Reads the image height in pixels
    move.l  HEADER_SKIP_HEIGHT(a0), d7
    jsr     ConvertEndianLong
    move.l  d7, (height)
    
    ; Reads the number of bits in the color that this image supports    
    ; Clears d7 to work with the lower word
    clr.l   d7
    move.w  HEADER_SKIP_BITS(a0), d7
    jsr     ConvertEndianWord
    move.w  d7, (bit_color_count)  
    
    ; Skip offset calculation in the case of 32 bit color   
    cmpi.w  #BIT_COLOR_32, d7
    beq     SkipOffset
    
    ; Throw an error if this isn't 24 or 32 bit 
    cmpi.w  #BIT_COLOR_24, d7
    bne     ErrorMessage

    ; Bitmasks to get the remainder after dividing by 4
    ; This is used to calculate how much padding we'll need at the end of the line of pixels 
    move.l  (width), d4
    andi.l  #3, d4
        
SkipOffset
    ; If we skipped the offset calculation step, we set this to 0, d4's default
    move.b  d4, (four_padding)

    ; Divide the bit count in d7 by 8 to get the bytes per pixel 
    lsr.w   #3, d7
        
    ; d5 holds the number of bytes from the edge of the image
    ; Width * bytes per pixel + padding 
    move.l  (width), d5
    mulu.w  d7, d5
    add.l   d4, d5
    move.l  d5, (bytes_per_row)
    
    rts
    

* Subroutine DrawChunk *
; Iterative subroutine to draw out the chunk of a bitmap 
; Includes the labels DrawChunk, ResetWidth, DrawChunkLoop, SetColor, and ContinueLoop
DrawChunk
    move.l  OFFSET_IMAGE_FILE(sp), a0

    move.w  (bit_color_count), d4
    lsr.w   #3, d4
        
    ; d5 holds the number of bytes from the edge of the image
    move.l  OFFSET_CHUNK_CORNER_X(sp), d5
    mulu.w  d4, d5
    
    ; d7 holds the total number of bytes per row
    move.l  (bytes_per_row), d7
    
    ; d6 holds the number of rows between the bottom of the image and the chunk
    move.l  (height), d6
    sub.l   OFFSET_CHUNK_CORNER_Y(sp), d6
    sub.l   OFFSET_CHUNK_HEIGHT(sp), d6
    
    ; Set a0 to the beginning of the chunk
    ; This calculates the number of rows and columns from the bottom corner
    move.l  OFFSET_IMAGE_FILE(sp), a0
    add.l   (pixel_array_offset), a0
    add.l   d5, a0
    mulu.w  d7, d6
    add.l   d6, a0
        
    ; Use d5 to see if we've exceeded the width
    move.l  OFFSET_CHUNK_CORNER_X(sp), d5
    add.l   OFFSET_CHUNK_WIDTH(sp), d5
    cmp.l   (width), d5 
    bgt     ErrorMessage
    
    ; Use d5 to see if we've exceeded the height
    move.l  OFFSET_CHUNK_CORNER_Y(sp), d5
    add.l   OFFSET_CHUNK_HEIGHT(sp), d5
    cmp.l   (height), d5 
    bgt     ErrorMessage    
    
    ; Save d5 as a height sentinel with the y coordinate boundary (the y coordinate of the furthest up pixel
    move.l  OFFSET_DISPLAY_CORNER_Y(sp), d5
    
    ; d2 will hold the y coordinates of the display pixel location
    move.l  OFFSET_DISPLAY_CORNER_Y(sp), d2
    add.l   OFFSET_CHUNK_HEIGHT(sp), d2
    
    ; d3 will hold the x coordinate boundary (the x coordinate of the furthest right pixel)    
    move.l   OFFSET_CHUNK_WIDTH(sp), d3
    
    ; Multiply this number of pixels by the pixel size in d4
    ; This is useful for later to calculate the "chunk skip"
    mulu.w  d3, d4
    
    ; d3 is finalized as a width sentinel by adding the corner offset
    add.l  OFFSET_DISPLAY_CORNER_X(sp), d3
    
    ; Now, d7 holds our "chunk skip"
    ; When we reach the end of a row in our chunk, we add d7
    ; This takes us to the correct pixel row and column for the next row of pixels
    sub.l   d4, d7
    
ResetWidth
    ; d4 holds our "current width" value
    ; Here, d4 is reset to the furthest left chunk value, indicating that we are printing a new row
    move.l  OFFSET_DISPLAY_CORNER_X(sp), d4
    
DrawChunkLoop
    ; Read the color on the current pixel 
    clr.l   d1
    
    ; Grabs the BGR values from the bitmap
    move.b  (a0)+, d1
    lsl.l    #8, d1    
    move.b  (a0)+, d1
    lsl.l    #8, d1         
    move.b  (a0)+, d1

    ; Checks if we need to jump over the alpha value (32 bit color)   
    ; alpha values will not be drawn over 
    ; BACKGROUND IMAGES SHOULD NOT HAVE ALPHA VALUES IN THEM!
    cmpi.w  #BIT_COLOR_24, (bit_color_count)
    beq     SetColor
    cmpi.b  #$FF, (a0)+
    beq     ContinueLoop
    
SetColor
    ; Set the color for the pixel 
    move.b  #TASK_PEN_COLOR,d0

    TRAP #15
    
    ; Draw the pixel at the current pixel's width and height           
    move.l  d4,d1
    move.b  #TASK_DRAW_PIXEL,d0
    TRAP #15  

ContinueLoop
    ; Increment the current width value and check if we're at the end of the row
    addi.l  #1, d4
    cmp.l   d3, d4
    blt     DrawChunkLoop
    
    ; When we've reached row's end, add the "chunk skip"
    add.l   d7, a0
        
    ; Decrement the current height value and check if we're at the last row
    subi.l  #1, d2
    cmp.l   d5, d2
    bgt     ResetWidth 

    ; By this point, the image is drawn
    ; Return to main
    rts


* Subroutine MovePlayer *
; Updates the square's position based on input
; Includes the labels GetInput, ReadSpace, ReadRight, ReadLeft, Fall, CheckTop, CheckLeft, CheckRight, CheckCollisions, RedrawPlayer, and Done
MovePlayer 
    ; Moves square position data into d4 and d5
    clr.l   d2


GetInput
    ; Reads input from the user 
    ; Skips if there's been no falling or key input 
    move.l  #KEY_SPLR_INPUT,d1
    move.b  #TASK_READ_INPUT,d0
    TRAP    #15
    or.l    d1, d2
    move.l  (air_time), d0
    or.l    d0, d2
    beq     Done
    cmp.l   #$000000, d1
    beq     Fall

ReadSpace
    ; Reads the space bar (first 2 hexadigits in SP - L - R)
    cmp.l   #0, d0
    bne     ReadRight
    move.l  d1, d3
    swap    d3
    cmp.b   #$FF, d3
    bne     ReadRight
    move.l  #$100, d0

ReadRight
    ; Reads the right arrow key (last 2 hexadigits in SP - L - R)
    cmp.w   #$00FF, d1
    bne     ReadLeft
    addi.w  #WALK_VELOCITY_VALUE, d4

ReadLeft
    ; Reads the left arrow key (middle 2 hexadigits in SP - L - R)
    cmp.w   #$FF00, d1
    bne     Fall
    subi.w  #WALK_VELOCITY_VALUE, d4

Fall
    ; Updates falling timescale  
    cmp.l   #0, d0
    beq     CheckTop
    move.l  d0, d1
    add.l   #TIMESCALE_VALUE, d0
    move.l  d0, (air_time)
    ; Calculates verticality based on jump velocity and gravity 
    ; Here, our timescale acts as a fixed point number with 8 fractional bits 
    mulu.w  d1, d1
    ; After squaring, we need to shift back by twice the usual number of fractional bits 
    lsr.l   #8, d1
    lsr.l   #8, d1
    mulu.w  #GRAVITY_ACCELERATION_VALUE, d1
    add.w   d1, d5
    sub.w   #JUMP_VELOCITY_VALUE, d5

CheckTop
    ; Check if we're out of bounds in the vertical sense 
    cmp.w   #UPPER_BOUND, d5
    bgt     CheckLeft
    move.w  #UPPER_BOUND, d5

CheckLeft  
    ; Check if we're out of bounds in the left side 
    cmp.w   #LEFT_BOUND, d4
    bgt     CheckRight   
    move.w  #LEFT_BOUND, d4

CheckRight    
    ; Check if we're out of bounds in the right side
    cmp.w   #RIGHT_BOUND, d4
    blt     CheckCollisions 
    move.w  #RIGHT_BOUND, d4 

CheckCollisions 
    ; Get the last known vertical position
    move.w  (square_position_last+2), d7
    
    ; Check for collisions with other game objects 
    jsr     PlatformCheck
    
    ; Get the last known horizontal position
    move.w  (square_position_last), d6

RedrawPlayer
    ; Save off registers 
    movem.l ALL_REG,-(sp)

    ; Adjust d7 to draw over the border left behind by the player 
    sub.w   #1, d7

    ; Draw over where the player used to be 
    move.l  d7, -(sp)           ; Top left corner of the display, y coordinate  SP offset = 7
    move.l  d6, -(sp)           ; Top left corner of the display, x coordinate  SP offset = 6
    move.l  #PLAYER_SIZE, -(sp)          ; Height of the chunk                           SP offset = 5
    move.l  #PLAYER_SIZE, -(sp)          ; Width of the chunk                            SP offset = 4
    move.l  d7, -(sp)           ; Top left corner of the chunk, y coordinate    SP offset = 3
    move.l  d6, -(sp)           ; Top left corner of the chunk, x coordinate    SP offset = 2
    move.l  #imageFile, -(sp)   ; Pointer to the image file                     SP offset = 1
    jsr     DrawChunk
    
    ; Fix the stack
    add.l   #RESET_STACK_OFFSET, sp      

    ; Restore registers 
    movem.l (sp)+,ALL_REG 
    
    ; Finally, draw the player in their new spot and update the buffer
    jsr     DrawPlayer
    move.l  #TASK_UPDATE_BUFFER, d0
    TRAP    #15

Done
    ; Save the new square position 
    move.w  d4, d6
    swap    d6
    move.w  d5, d6

    move.l  d6, (square_position_last)
    rts


* Subroutine PlatformCheck *
; Includes the labels PlatformCheckRecurse, SlideCheck, PlatformCheckStep, and PlatformReturn
PlatformCheck
    ; Clear out some of the registers for operation
    ; d6 checks if we've collided with the target sigil
    clr.l   d6
    
    ; d0 tracks if there's been a collision or not
    move.w  #0, d0
    
    ; d2-5 hold player coordinates
    move.w  d4, d2
    add.w   #PLAYER_SIZE, d2
    move.w  d5, d3
    add.w   #PLAYER_SIZE, d3
    
    ; Load in our collision array
    lea     collisions_array, a0
    lea     end_collisions_array, a1
    
PlatformCheckRecurse
    ; Check if the top and bottom bounds of the player are within the collider object
    cmp.w   2(a0), d5
    bgt     PlatformCheckStep
    cmp.w   2(a0), d3
    blt     PlatformCheckStep
    
    ; If we're vertically colliding, check if we're horizontally colliding
    cmp.w   (a0), d2
    blt     PlatformCheckStep
    cmp.w   4(a0), d4
    bgt     PlatformCheckStep
    
    ; See if we were above the collider in the previous frame
    ; If not, no need to check for collisions any further
    cmp.w   d7, d5 
    blt     PlatformCheckStep
    
    ; Position the player on top of the object instead of through it 
    move.w  2(a0), d5
    add.w   #1, d0
    sub.w   #PLAYER_SIZE, d5
    
    ; If there's something below, stop falling
    cmp.l   #BEGIN_FALL, (air_time)
    blt     SigilCollideCheck
    move.l  #0, (air_time)
    
SigilCollideCheck
    ; See if we ran into the target sigil
    move.b  (selected_sigil), d6
    cmp.w   8(a0),  d6
    bne     PlatformCheckStep
    
    ; If we did, give ourselves a point and pick a new sigil
    jsr     Score
    jsr     PickSigil
    
PlatformCheckStep
    ; Go to the next collider
    add.l   #COLLIDER_INFO_SIZE, a0
    cmp.w   a0, a1
    bne     PlatformCheckRecurse
    
    ; At the end of the collider list, if there are no collisions, start falling
    cmp.w   #0, d0
    bne     PlatformReturn
    cmp.l   #0, (air_time)
    bne     PlatformReturn
    add.l   #BEGIN_FALL, (air_time)
    
PlatformReturn
    rts


* Subroutine WarmRandomEngine *
; Runs through the RNG a few times to diversify output 
; Programmed by Tom Carbone    
WarmRandomEngine
    move.b  #TASK_GET_TIME,d0
    TRAP    #15
    move.b  d1,d2
    eor.l   #$AA,d2
    lsl.l   #8,d1
    move.b  d2,d1
        
    move.l  d1,random_val
    jsr     SeedRandomNumber
    jsr     GetRandomByteIntoD6
    jsr     GetRandomByteIntoD6
    jsr     GetRandomByteIntoD6
    jsr     GetRandomByteIntoD6
    jsr     GetRandomByteIntoD6
    jsr     GetRandomByteIntoD6
    jsr     GetRandomByteIntoD6
    jsr     GetRandomByteIntoD6

    rts


* Subroutine SeedRandomNumber *
; Seeds randomness with the system clock 
; Programmed by Tom Carbone    
SeedRandomNumber
    movem.l ALL_REG,-(sp)
    clr.l   d6
    move.b  #TASK_GET_TIME,d0
    TRAP    #15

    move.l  d1,random_val
    movem.l (sp)+,ALL_REG
    rts


* Subroutine GetRandomByteIntoD6 *
; Gets a random byte into d6
; Programmed by Tom Carbone    
GetRandomByteIntoD6
    movem.l d0,-(sp)
    movem.l d1,-(sp)
    movem.l d2,-(sp)
    move.l  random_val,d0
    moveq	#$AF-$100,d1
    moveq	#18,d2
Ninc0	
	add.l	d0,d0
	bcc	Ninc1
	eor.b	d1,d0
Ninc1
	dbf	d2,Ninc0
	
	move.l	d0,random_val
	
	move.l	d0,d6
	
    movem.l (sp)+,d2
    movem.l (sp)+,d1
    movem.l (sp)+,d0
    rts
        

* Subroutine GetRandomLongIntoD6 *
; Gets a random long word into d6
; Programmed by Tom Carbone        
GetRandomLongIntoD6
    movem.l ALL_REG,-(sp)
    jsr     GetRandomByteIntoD6
    move.b  d6,d5
    jsr     GetRandomByteIntoD6
    lsl.l   #8,d5
    move.b  d6,d5
    jsr     GetRandomByteIntoD6
    lsl.l   #8,d5
    move.b  d6,d5
    jsr     GetRandomByteIntoD6
    lsl.l   #8,d5
    move.b  d6,d5
    move.l  d5,temp_random_long
    movem.l (sp)+,ALL_REG
    move.l  temp_random_long,d6
    rts


* Subroutine DrawPlayer * 
; Draw the player at their current location
DrawPlayer
    ; Save off registers
    movem.l ALL_REG,-(sp)
    
    ; Clear space for the registers
    clr.l   d1
    clr.l   d2
    clr.l   d3

    ; Get the outline and fill colors
    move.l  #PLAYER_OUTLINE_COLOR, d1
    move.b  #TASK_PEN_COLOR, d0
    TRAP    #15    
    move.l  #PLAYER_FILL_COLOR, d1
    move.b  #TASK_FILL_COLOR, d0
    TRAP    #15

    ; Move the data on the square's position in to the registers
    move.w  d4, d1
    move.w  d5, d2
    move.w  d1, d3
    add.w   #PLAYER_SIZE-1, d3
    move.w  d2, d4
    add.w   #PLAYER_SIZE-1, d4
    
    ; Draw the squares
    move.b  #TASK_DRAW_SQUARE,d0
    TRAP    #15
    
    ; Fix registers and return 
    movem.l (sp)+,ALL_REG
    rts      


* Subroutine PickSigil *
; Randomly selects and displays the next place the player needs to go to
; Includes the label RandomSigil
PickSigil
    ; Save registers
    movem.l ALL_REG,-(sp)
    
RandomSigil
    ; Get a random byte
    clr.l   d6
    jsr     GetRandomByteIntoD6
    lsr.l   #8, d6
    lsr.l   #8, d6
    
    ; Modulo by the sigil count to get the current sigil
    divu.w  #SIGIL_COUNT, d6
    clr.w   d6
    swap    d6
    
    ; If we rechoose the current sigil, pick a new one!
    cmp.b   (selected_sigil), d6
    beq     RandomSigil
    
    ; Save it 
    move.b  d6, (selected_sigil)
    mulu.w  #SIGIL_SIZE, d6
    
    ; Draw the sigil
    move.l  #PARAMETER_SIGIL_DISPLAY_CORNER_Y, -(sp)  ; Top left corner of the display, y coordinate  SP offset = 7
    move.l  #PARAMETER_SIGIL_DISPLAY_CORNER_X, -(sp)  ; Top left corner of the display, x coordinate  SP offset = 6
    move.l  #PARAMETER_SIGIL_HEIGHT, -(sp)      ; Height of the chunk                           SP offset = 5
    move.l  #PARAMETER_SIGIL_WIDTH, -(sp)       ; Width of the chunk                            SP offset = 4
    move.l  #PARAMETER_SIGIL_CORNER_Y , -(sp)   ; Top left corner of the chunk, y coordinate    SP offset = 3
    move.l  d6, -(sp)    ; Top left corner of the chunk, x coordinate    SP offset = 2
    move.l  #imageFile, -(sp)                   ; Pointer to the image file                     SP offset = 1
    
    jsr     DrawChunk
    
    ; Fix stack and registers
    add.l   #RESET_STACK_OFFSET, sp    
    movem.l (sp)+,ALL_REG    
    
    rts


* Subroutine ConvertNumberToSeg *
; Converts number in d7 to segment
; Includes the labels ConvertNextDigit, ConvertDigitCheck, PaintItBlack, DrawCurrentSegment, and ConvertNumberToSegReset
ConvertNumberToSeg
    ; Load in the end of the segment array and the starting horizontal position
    lea     seven_seg_end, a6
    move.l  #SEG_STARTING_HORIZONTAL, d5

ConvertNextDigit
    ; Move the starting point left a bit for the current digit
    sub.l   #SEG_HORIZONTAL_OFFSET, d5
    
    ; Load up more array locations for navigation
    lea     seven_seg_line_array, a5
    lea     number_to_seg, a4
    
    ; Some basic arithmetic to get the current digit
    move.l  d7, d6
    divu    #10, d6
    swap    d6
    ext.l   d6
    add.l   d6, a4
    move.b  (a4), d6
    
ConvertDigitCheck
    ; Checks if the current segment for the current digit is colored in 
    move.l  d6, d1
    andi.l  #1, d1
    beq     PaintItBlack   
    
    ; Fill in red if the segment is colored in 
    move.l  #80, d0
    move.l  #SEG_COLOR_RED, d1
    TRAP    #15 
    bra     DrawCurrentSegment
    
PaintItBlack   
    ; Branch here if you see a red line and you want it painted black
    move.l  #80, d0
    move.l  #SEG_COLOR_BLACK, d1
    TRAP    #15 
    
DrawCurrentSegment
    ; Draw that segment
    jsr     DrawSeg 
    
ConvertNumberToSegReset
    ; Go to the next of the seven segments
    ; If we're at the end of the segments, on to the next digit
    add.l   #8, a5
    lsr.l   #1, d6
    cmp.l   a5, a6
    bne     ConvertDigitCheck
    
    ; If we're at the end of the digits, return out 
    divu    #10, d7
    ext.l   d7
    bne     ConvertNextDigit
    rts


* Subroutine DrawSeg *
; Draws the current segment
DrawSeg  
    ; Access the start and end points of the line from memory
    move.w  (a5),  d1
    move.w  2(a5), d2
    move.w  4(a5), d3
    move.w  6(a5), d4
    
    ; Adjust for scaling and offset
    lsl.l   #SEG_SCALING, d1
    add.l   d5, d1
    lsl.l   #SEG_SCALING, d2
    add.l   #SEG_VERTICAL_OFFSET, d2
    lsl.l   #SEG_SCALING, d3
    add.l   d5, d3
    lsl.l   #SEG_SCALING, d4
    add.l   #SEG_VERTICAL_OFFSET, d4
    
    ; Draw the line
    move.l  #TASK_DRAW_LINE, d0
    trap    #15
    
    rts 


* Subroutine Score *
; Increments the score.
; Includes the labels BeginConversion and ScoreReturn
Score
    ; Save off registers
    movem.l ALL_REG,-(sp)
    
    ; Increment the score, but bound it by the max
    move.l  (current_score), d7
    add.l   #SCORE_VALUE, d7
    cmp.l   #MAXIMUM_SCORE, d7
    ble     BeginConversion
    move.l  #MAXIMUM_SCORE, d7
        
BeginConversion    
    ; Save current score and convert to the seven segment display
    move.l  d7, (current_score)
    jsr     ConvertNumberToSeg

    ; Check if we won!
    cmp.l   #MAXIMUM_SCORE, (current_score)
    beq     Over
 
ScoreReturn
    ; Restore registers
    movem.l (sp)+,ALL_REG 
    rts
    

* Subroutine ErrorMessage *
; Puts out a generic error message.
ErrorMessage
    lea (error_message), a1
    move.l  #TASK_PRINT_MESSAGE,d0
    TRAP #15
    
    move.l  #TASK_UPDATE_BUFFER, d0
    TRAP    #15       
    
    move.l  #$9,d0
    TRAP #15
    

* Routine START *   
; The "main" of our program
; Includes the labels GameLoop, NoInput, Over, LoadLose, LoadWin, and End
START   ORG    $1000

GameStart
    ; Sets resolution of the output image
    move.b  #TASK_OUTPUT_RESOLUTION,d0  
    move.l  #OUTPUT_RESOLUTION_WIDTH,d1
    swap.w  d1
    move.w  #OUTPUT_RESOLUTION_HEIGHT,d1
    TRAP    #15 

    ; Puts us in draw buffer mode 
    move.l  #TASK_SET_DRAW_MODE, d0
    move.l  #TASK_DRAW_MODE_BUFFER, d1
    TRAP    #15
    
    ; Reads the BMP's data
    movem.l ALL_REG,-(sp)
    lea     imageFile, a0
    jsr     InitBackgroundMetadata
    movem.l (sp)+,ALL_REG 

    ; Draw the background 
    movem.l ALL_REG,-(sp)
    
    ; Load the arguments for the subroutining chunk onto the stack
    move.l  #PARAMETER_DISPLAY_CORNER_Y, -(sp)  ; Top left corner of the display, y coordinate  SP offset = 7
    move.l  #PARAMETER_DISPLAY_CORNER_X, -(sp)  ; Top left corner of the display, x coordinate  SP offset = 6
    move.l  #PARAMETER_CHUNK_HEIGHT, -(sp)      ; Height of the chunk                           SP offset = 5
    move.l  #PARAMETER_CHUNK_WIDTH, -(sp)       ; Width of the chunk                            SP offset = 4
    move.l  #PARAMETER_CHUNK_CORNER_Y , -(sp)   ; Top left corner of the chunk, y coordinate    SP offset = 3
    move.l  #PARAMETER_CHUNK_CORNER_X, -(sp)    ; Top left corner of the chunk, x coordinate    SP offset = 2
    move.l  #imageFile, -(sp)                   ; Pointer to the image file                     SP offset = 1
    
    ; Call the draw chunk method
    jsr     DrawChunk
    add.l   #RESET_STACK_OFFSET, sp    
    movem.l (sp)+,ALL_REG 
        
    ; Warm up the random number generator 
    jsr     WarmRandomEngine

    ; Randomly select a target sigil
    jsr     PickSigil

    ; Initialize the score to 0
    clr.l   d7 
    jsr     ConvertNumberToSeg      
    
    ; Clean slate for our registers 
    clr.l   d0
    clr.l   d1
    clr.l   d2
    clr.l   d3
    clr.l   d4
    clr.l   d5
    clr.l   d6
    clr.l   d7 
    
    ; Save starting time    
    move.b  #TASK_GET_TIME,d0
    TRAP    #15    
    
    ; Get into seconds and save in d2
    divu    #100, d1
    ext.l   d1
    move.l  d1, (start_time)
            
    move.w  (square_position), d4
    move.w  (square_position+2), d5
    
    jsr DrawPlayer
        
    ; Update from the buffer 
    move.l  #TASK_UPDATE_BUFFER, d0
    TRAP    #15           
            
GameLoop
    ; We only check for input every 65536 frames
    cmp.w   #0, d0
    bne     NoInput
    jsr     MovePlayer       

NoInput
    add.w   #1, d0    
    bra     GameLoop
    
Over
    ; We go here when the game is over
    move.b  #TASK_FILL_COLOR, d0
    move.l  #$0, d1
    TRAP #15 
      
LoadWin
    ; Print win message
    lea (win_message), a1
    move.l  #TASK_PRINT_MESSAGE,d0
    TRAP #15
    
    ; Track how long it took the player to win
    move.b  #TASK_GET_TIME,d0
    TRAP    #15    
    divu    #100, d1
    ext.l   d1
    sub.l   (start_time), d1
    
    ; Display the player's record
    move.b  #TASK_PRINT_NUMBER, d0
    TRAP #15

End
    ; Finally update the buffer 
    move.l  #TASK_UPDATE_BUFFER, d0
    TRAP    #15       

RestartGame

    ; Space restarts the game 
    move.l  #KEY_SPLR_INPUT,d1
    move.b  #TASK_READ_INPUT,d0
    TRAP    #15
    cmp.l   #$000000, d1
    beq     RestartGame
    swap    d1
    cmp.b   #$FF, d1
    bne     RestartGame
    
    ; Reset the score and position    
    move.w  #PLAYER_POSITION, (square_position)
    move.w  #PLAYER_POSITION, (square_position+2)
    move.l  (square_position), (square_position_last)
    move.l  #$01000000, a7
    
    move.l  #0, (current_score)
    
    ; Clear screen and reset it 
    move.b  #CLEAR_SCREEN_VALUE, d0 
    move.w  #RESET_SCREEN, d1
    trap    #15
    
    bra     GameStart

    move.b  #9,d0
    TRAP    #15 
        

* Variables for the program *

; Current position of the player
square_position         dc.w    250, 250
; Last position of the player
square_position_last    dc.w    250, 250

; Coordinates of the objects that the player can collide with 
collisions_array        dc.w    51, 128, 125, 151,  0   ; top left platform
                        dc.w    0, 300, 104, 384,   1   ; left floor triangle
                        dc.w    206, 224, 306, 247, 2   ; middle platform
                        dc.w    386, 128, 460, 151, 4   ; top right platform
                        dc.w    408, 300, 512, 384, 3   ; right floor triangle
                        dc.w    105, 300, 407, 384, 5   ; floor
end_collisions_array

; The currently displayed sigil that the player has to go touch 
selected_sigil      ds.b 1

; The player's current score
current_score        dc.l 0

; When the player started
start_time      dc.l    0

; Where the array of pixels actually begins
pixel_array_offset  ds.l 1

; Width of the image in pixels
width               ds.l 1

; Height of the image in pixels
height              ds.l 1

; Number of bits in the color (24 or 32)
bit_color_count     ds.w 1

; Value we add to pad the end of a line of pixels 
; Used for calculating divisibility by 4 
; Though the value is always less than 4, four_padding is stored as a word to avoid addressing issues
four_padding        ds.w 1

; Stores the number of bytes in a row of pixels in our image
bytes_per_row       ds.l 1

; Generic error message.
error_message       dc.b 'An error occurred, please see documentation in the header of the source file!',0
win_message       dc.b 'You won! (Press Space to Restart) It took you this many seconds: ',0

; How many "frames" the players has been in the air 
air_time            dc.l    0

; Input countdown
no_input            ds.w    1

; Variables used for random number generation
; Pulled from Tom Carbone's example code
random_val          ds.l    1
temp_random_long    ds.l    1



; Coordinates of the segments in a seven segment display
seven_seg_line_array
                        dc.w    0,1,1,1 ;g
                        dc.w    0,0,0,1 ;f
                        dc.w    0,1,0,2 ;e
                        dc.w    0,2,1,2 ;d
                        dc.w    1,1,1,2 ;c
                        dc.w    1,0,1,1 ;b
                        dc.w    0,0,1,0 ;a
seven_seg_end                        

; Array of numbers whose bit values represent on-off switches for the seven segment display
number_to_seg           dc.b    $7E     ;0
                        dc.b    $30     ;1
                        dc.b    $6D     ;2
                        dc.b    $79     ;3
                        dc.b    $33     ;4
                        dc.b    $5B     ;5
                        dc.b    $5F     ;6
                        dc.b    $70     ;7
                        dc.b    $7F     ;8
                        dc.b    $7B     ;9
    
; Beginning of the background image file 
; Change this file name to change the source image
imageFile   INCBIN  "tempmap.bmp"

    END    START        ; last line of source






*~Font name~Courier New~
*~Font size~12~
*~Tab type~1~
*~Tab size~4~
